// AIQDoc.cpp : Implementierung der Klasse CAIQDoc
// Beispielcode zur Implementierung diverser Wegfindungsroutinen
// Implementiert sind:
// - Gerade Linie
// - Gnstigster nchster Schritt
// - Dijkstra
// - A*
// Zur freien Verwendung freigegeben, bitte Ursprung mitangeben
// 18. Juni 2001
// Ralph Meier, Bergstr. 4, 58739 Wickede (Ruhr)
// email: Ralph.Meier@FernUni-Hagen.de


#include "stdafx.h"
#include "AIQ.h"
#include "AIQDoc.h"
#include <afxtempl.h>

#ifdef _DEBUG
#define new DEBUG_NEW
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

/////////////////////////////////////////////////////////////////////////////
// CAIQDoc

IMPLEMENT_DYNCREATE(CAIQDoc, CDocument)

BEGIN_MESSAGE_MAP(CAIQDoc, CDocument)
	//{{AFX_MSG_MAP(CAIQDoc)
	ON_COMMAND(ID_WEGFINDUNG_A, OnWegfindungAStar)
	ON_COMMAND(ID_WEGFINDUNG_BESTERSCHRITT, OnWegfindungBesterschritt)
	ON_COMMAND(ID_WEGFINDUNG_DIJKSTRA, OnWegfindungDijkstra)
	ON_COMMAND(ID_WEGFINDUNG_LUFTLINIE, OnWegfindungLuftlinie)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

/////////////////////////////////////////////////////////////////////////////
// CAIQDoc Konstruktion/Destruktion

CAIQDoc::CAIQDoc()
{
	// Initialisierung der leeren Welt ohne Start und Ziel
	Start = CPoint(-1, -1);	
	Target = CPoint(-1, -1);

	for (int x=0 ; x < WORLDX; x++)
		for (int y=0; y < WORLDY; y++)
			World[x][y] = 4;			// mittlere Hhe
}

CAIQDoc::~CAIQDoc()
{
}

BOOL CAIQDoc::OnNewDocument()
{
	if (!CDocument::OnNewDocument())
		return FALSE;

	// Reset the World (should be real sometimes ;-)
	Start = CPoint(-1, -1);	
	Target = CPoint(-1, -1);

	for (int x=0 ; x < WORLDX; x++)
		for (int y=0; y < WORLDY; y++)
			World[x][y] = 4;			// mittlere Hhe

	return TRUE;
}



/////////////////////////////////////////////////////////////////////////////
// CAIQDoc Serialisierung

void CAIQDoc::Serialize(CArchive& ar)
{
	if (ar.IsStoring())
	{
		// ZU ERLEDIGEN: Hier Code zum Speichern einfgen
	}
	else
	{
		// ZU ERLEDIGEN: Hier Code zum Laden einfgen
	}
}

/////////////////////////////////////////////////////////////////////////////
// CAIQDoc Diagnose

#ifdef _DEBUG
void CAIQDoc::AssertValid() const
{
	CDocument::AssertValid();
}

void CAIQDoc::Dump(CDumpContext& dc) const
{
	CDocument::Dump(dc);
}
#endif //_DEBUG

/////////////////////////////////////////////////////////////////////////////
// CAIQDoc Befehle

void CAIQDoc::RaiseWorld(int x, int y)
{
	// Bei Klick mit der linken Maustaste wird das angeklickte
	// Weltfeld auf eine hhere (Kosten-)Stufe gehoben
	World[x][y]<8 ? (World[x][y])++ : World[x][y];
	return;
}

void CAIQDoc::LowerWorld(int x, int y)
{
	// Bei Klick mit der rechten Maustaste wird das angeklickte
	// Weltfeld auf eine niedrigere (Kosten-)Stufe gesenkt
	World[x][y]>0 ? (World[x][y])-- : World[x][y];
	return;
}

bool CAIQDoc::OnWegfindungAStar() 
{
	// A* - Algorithmus
	// Rckgabewert: TRUE, wenn Weg gefunden; FALSE, wenn kein Weg gefunden
	// Quelle: http://www.in.tum.de/~muellemi/amr/report/node17.html

	// Deklaration der bentigten Variablen
	CList<CPoint,CPoint&> QSuche;			// open-list
	CList<CPoint,CPoint&> QErledigt;

	long QDistanz[WORLDX][WORLDY];
	long QHeuristik[WORLDX][WORLDY];
	long QSumme[WORLDX][WORLDY];
	CPoint QVorgaenger[WORLDX][WORLDY];

	// Array zur Vereinfachung der Berechnung der Nachbarpositionen
	static CPoint neighbours[4] =
	{
		{CPoint(-1,0)}, {CPoint(0,-1)}, 
		{CPoint(1,0)}, {CPoint(0,1)}
	};

	// Initialisierung
	
	QSuche.AddHead(Start);
	for (int y=0; y < WORLDY; y++)
		for (int x=0; x < WORLDX; x++)
		{
			QDistanz[x][y] = 999999;
			QHeuristik[x][y] = (Target.x - x) + (Target.y - y);		// geschtzte Enfernung zum Ziel
			QVorgaenger[x][y] = CPoint(-1, -1);						// kein Vorgnger
		}

	// Vorbelegungen fr Start und Ziel
	QDistanz[Start.x][Start.y] = 0;
	QHeuristik[Target.x][Target.y] = 0;
	QVorgaenger[Start.x][Start.y] = CPoint(-1, -1);
	QSumme[Start.x][Start.y] = QDistanz[Start.x][Start.y] + QHeuristik[Start.x][Start.y];

	// Suche
	while (!(QSuche.IsEmpty()))
	{
		POSITION pos, min;		// Iterator fr Liste, Zeiger auf Element mit minimalen Wegkosten
		pos = min = QSuche.GetHeadPosition();
		long minwert = 999999;
		// Ermittle das Element u aus QSuche mit minimaler QSumme[u]
		CPoint u = QSuche.GetAt(pos);
		while (pos != NULL)
		{
			if (QSumme[u.x][u.y] < minwert)
			{
				min = pos;
				TRACE("Minimum bei Pos %d\n",min);
				minwert = QSumme[u.x][u.y];
			}
			u = QSuche.GetNext(pos);
		}
		TRACE("Minimum: %d %d  Pos: %d",u.x,u.y,min);
		u = QSuche.GetAt(min);
		// Lsche u aus QSuche
		QSuche.RemoveAt(min);

		// Ziel noch nicht erreicht?
		if (u != Target)
		{
			TRACE ("u: %d %d   - Target: %d %d\n",u.x,u.y,Target.x,Target.y);
			// Prfe auf neue krzeste Wege und aktualisiere Vorgnger und QDistanz
			for (int i=0; i<4; i++)
			{
				CPoint v = CPoint(u.x + neighbours[i].x, u.y + neighbours[i].y);
				// berschreitung der Randzonen vermeiden
				if ((v.x >= 0) && (v.x < WORLDX) &&
					(v.y >= 0) && (v.y < WORLDY))
				{
					int distanz = 1 + abs(World[u.x][u.y] - World[v.x][v.y]);
				
					if ((QDistanz[v.x][v.y] > QDistanz[u.x][u.y] + distanz) 
						 || ((!QSuche.Find(v)) && (!QErledigt.Find(v))))
					{
						QDistanz[v.x][v.y] = QDistanz[u.x][u.y] + distanz;
						QSumme[v.x][v.y] = QDistanz[v.x][v.y] + QHeuristik[v.x][v.y];
						QVorgaenger[v.x][v.y] = u;
	
						// Streiche v aus QSuche, wenn vorhanden
						pos = QErledigt.Find(v);
						if (pos != NULL)	QErledigt.RemoveAt(pos);

						// Fge v in QSuche, wenn noch nicht vorhanden
						pos = QSuche.Find(v);
						if (pos == NULL)	QSuche.AddHead(v);
					}
				}
			}
			// Den untersuchten Knoten u in die Liste der erledigten Knoten bernehmen
			QErledigt.AddHead(u);
		}
		else
		{
			// Weg wurde gefunden, QSuche leeren (Abbruchkriterium und aufrumen)
			QSuche.RemoveAll();
			// Weg rckwrts ausgeben
			CPoint u = Target;
			while (u != Start)
			{
				World[u.x][u.y] = 99;
				u = QVorgaenger[u.x][u.y];
			}

			// View aktualisieren
			UpdateAllViews(NULL);

			return TRUE;
		}
	}
	// Hier Fehlermeldung einfgen: Es konnte kein Weg gefunden werden
	return FALSE;
}

bool CAIQDoc::OnWegfindungBesterschritt() 
{
	// Whlt unter den Schritten in Richtung auf das Ziel den
	// gnstigsten Schritt aus
	
	// Rckgabewert: TRUE, wenn Weg gefunden; FALSE, wenn kein Weg gefunden

	CPoint pos = Start;
	long costs = 0;

	while (pos != Target)
	{
		int dx = Target.x - pos.x;
		int dy = Target.y - pos.y;
	
		// dx und dy normieren
		dx > 0 ? dx = 1 : (dx < 0 ? dx = -1 : dx = 0);
		dy > 0 ? dy = 1 : (dy < 0 ? dy = -1 : dy = 0);

		// berechne Richtung mit minimalen Bewegungskosten
		// nur horizontale und vertikale Bewegungsmglichkeit
		BYTE mincost = World[pos.x+dx][pos.y];
		int dirX = dx;
		int dirY = 0;
		TRACE("\nDas Minimum von %d und %d ist ",mincost,World[pos.x][pos.y+dy]);
		//mincost = __min(mincost,World[pos.x][pos.y+dy]);
		TRACE("%d\n",mincost);
		if (mincost > World[pos.x][pos.y + dy])
		{
			mincost = World[pos.x][pos.y + dy];
			dirX = 0;
			dirY = dy;
		}
		// Bewegungskosten berechnen
		costs += 1 + abs(mincost - World[pos.x][pos.y]);
		
		// Nchster Schritt
		pos.x += dirX;
		pos.y += dirY;

		// Weg einzeichnen
		World[pos.x][pos.y] = 99;
	}
	// View aktualisieren
	UpdateAllViews(NULL);

	// Weg gefunden
	return TRUE;
}

void CAIQDoc::OnWegfindungDijkstra() 
{
	// Wegfindung mittels Dijkstra-Algorithmus
	// Rckgabewert: TRUE, wenn Weg gefunden; FALSE, wenn kein Weg gefunden

	// Quelle: www.irf.de/seminar/schlette/stdt_1.htm

	// Deklaration diverser notwendiger Arrays
	long QDistanz[WORLDX][WORLDY];
	BOOL QErledigt[WORLDX][WORLDY];
	// BYTE QSuche[WORLDX][WORLDY];
	CPoint QVorgaenger[WORLDX][WORLDY];

	// Array zur Vereinfachung der Berechnung der Nachbarpositionen
	static int neighbours[4][2] =
	{
		{-1,0}, {0, -1}, {1, 0}, {0, 1}
	};


	// Initialisierung des Dijkstra-Arrays mit allen Knoten
	for (int y=0; y< WORLDY-1; y++)
		for (int x=0; x< WORLDX; x++)
		{
			// Welt-"Knoten" in das Dijkstra-Array aufnehmen
			QErledigt[x][y] = FALSE;
			QDistanz[x][y] = 999999;
			QVorgaenger[x][y] = CPoint(-1,-1);
		}
	// Vorbelegung fr Start-Feld
	QDistanz[Start.x][Start.y] = 0;

	// Betrachte alle Nachbarknoten des Startknoten und fge Kosten zu
	// diesen in Q ein
	for (int i=0; i<4; i++)
	{
		int nx = Start.x + neighbours[i][0];
		int ny = Start.y + neighbours[i][1];

		if (nx < 0) nx = 0;
		if (nx > WORLDX) nx = WORLDX;
		if (ny < 0) ny = 0;
		if (ny > WORLDY) ny = WORLDY;

		QDistanz[nx][ny] = (World[nx][ny] - World[Start.x][Start.y] + 1);
		QVorgaenger[nx][ny] = Start;
	}
	// Vorgnger des Startknotens ist inexistent
	CPoint vorg = CPoint(-1,-1);

	// Die Suche beginnt
	for (i=0; i<(WORLDX * WORLDY); i++)
	{
		// Ermittle den Index des (noch nicht erledigten) Feldelements v mit minimalen Wegekosten
		CPoint min = CPoint(-1, -1);
	    long minwert = 999999;
		for (y=0; y< WORLDY; y++)
			for (int x=0; x < WORLDX; x++) 
			{
				if (!(QErledigt[x][y]))
				{
					// TRACE("x: %d y: %d - min: %d %d\n",x,y,min.x,min.y);
					long wert = QDistanz[x][y];
					if (wert < minwert)
					{
						min.x = x;
						min.y = y;
						TRACE("\nwert: %d minwert: %d, min: %d %d\n", wert,minwert,min.x,min.y);
						minwert = QDistanz[min.x][min.y];
						TRACE("minwert neu gesetzt\n");
					}
				}
			}
		// Markiere Feldelement v als erledigt
		QErledigt[min.x][min.y] = TRUE;

		// betrachte alle noch nicht erledigten Nachbarn von v und aktualisiere evt.
		// deren krzesten Pfad
		for (int n=0; n<4; n++)
		{
			int nx = min.x + neighbours[n][0];
			int ny = min.y + neighbours[n][1];

			if (nx < 0) nx = 0;
			if (nx > WORLDX-1) nx = WORLDX-1;
			if (ny < 0) ny = 0;
			if (ny > WORLDY-1) ny = WORLDY-1;

			TRACE("nx: %d ny: %d\n",nx,ny);
			// Entfernung zwischen v und dem Nachbarn berechnen
			BYTE Dist = (World[nx][ny] - World[min.x][min.y] + 1);
			if (!(QErledigt[nx][ny]))
			{
				TRACE("Untersuche unerledigten Nachbarn %d %d\n",nx,ny);
				// Neuer, krzerer Weg zum Nachbarknoten?
				if (QDistanz[nx][ny] > QDistanz[min.x][min.y] + Dist)
				{
					QDistanz[nx][ny] = QDistanz[min.x][min.y] + Dist;
					// speicher Vater zu v
					QVorgaenger[nx][ny] = min;
				}
			}
		}
	}

	TRACE("\nWege wurden berechnet\n");

	// Ausgabe des krzesten Weges (wird rckwrts gezeichnet)
	// Korrekter Weg sollte entweder durch Austausch von Start und Ziel
	// oder durch Zwischenspeicherung in einem LIFO-Speicher/Stapel
	// ausgegeben werden
	CPoint pos = Target;
	do
	{
		TRACE("pos: %d %d\n",pos.x,pos.y);
		pos = QVorgaenger[pos.x][pos.y];
		World[pos.x][pos.y] = 99;
	} while (pos != Start);

	// View aktualisieren
	UpdateAllViews(NULL);	

	// Weg gefunden
	return TRUE;
}


void CAIQDoc::OnWegfindungLuftlinie() 
{
	// Rasterlinienalgorithmus nach Bresenham
	// Rckgabewert: TRUE, wenn Weg gefunden; FALSE, wenn kein Weg gefunden

	// Direkter Weg von Start nach Ziel
	// ohne Bercksichtigung von Wegekosten
	CPoint Pos = Start;				// Zwischenspeicher fr aktuelle Position zwischen Start und Ziel
	
	int costs = 0;					// Wegekosten
	int dx = Target.x - Start.x;	// x-Differenz zwischen Start und Ziel
	int dy = Target.y - Start.y;	// y-Differenz zwischen Start und Ziel
	int xincr = 0;					// bestimmt x-Richtung von Start nach Ziel (links/rechts)
	int yincr = 0;					// bestimmt y-Richtung von Start nach Ziel (oben/unten)
	int error;
	
	// Befindet sich das Ziel links vom Start,
	// muss nach links gewandert werden, sonst nach rechts
	dx<0 ? xincr = -1 : xincr = 1;

	// Befindet sich das Ziel oberhalb vom Start,
	// muss nach oben gewandert werden, sonst nach unten
	dy<0 ? yincr = -1 : yincr = 1;

	if (abs(dx) > abs(dy))
	{
		// x ist treibende Kraft
		// wir befinden uns in Oktand 1, 4, 5 oder 8
		error = -abs(dx) / 2;
		do
		{
			// Aktuelle Hhe merken fr Berechnung der Wegekosten
			BYTE oldheight = World[Pos.x][Pos.y];
			// Zeichne Wegstck
			World[Pos.x][Pos.y] = 99;
	
			// Muss y gendert werden?
			error += abs(dy);
			if (error>= 0)
			{
				Pos.y += yincr;
				error -= abs(dx);
			}
			Pos.x += xincr;
			// Berechne Wegekosten
			costs = costs + 1 + abs(World[Pos.x][Pos.y] - oldheight);	
		} while (Pos.x != Target.x);	// Ziel erreicht?
	}
	else
	{
		// y ist treibende Kraft
		// wir befinden uns in Oktand 2, 3, 6 oder 7
		error = - abs(dy) / 2;
		do
		{
			// Aktuelle Hhe merken fr Berechnung der Wegekosten
			BYTE oldheight = World[Pos.x][Pos.y];
			// Zeichne Wegstck
			World[Pos.x][Pos.y] = 99;

			error += abs(dx);
			if (error>= 0)
			{
				Pos.x += xincr;
				error -= abs(dy);
			}
			Pos.y += yincr;
			// Berechne Wegekosten
			costs = costs + 1 + abs(World[Pos.x][Pos.y] - oldheight);	
		} while (Pos.y != Target.y);	// Ziel erreicht?

	}
	// View aktualisieren
	UpdateAllViews(NULL);

	// Weg gefunden
	return TRUE;
}